<?php

namespace App\Http\Controllers\Admin\Kependudukan;

use App\Http\Controllers\Controller;
use App\Models\Kk;
use App\Models\Penduduk;
use App\Models\Profil;
use App\Models\RefSK;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use RealRashid\SweetAlert\Facades\Alert;
use Yajra\DataTables\Facades\DataTables;

class PendudukController extends Controller
{
    public function index(){
        $level = Auth::user()->level;
        $profil = Profil::where('pt_id', 1)->first();
        $refsks = RefSK::all();
        $kks = Kk::query()
                ->join('penduduks', 'kks.kepala_kk', '=', 'penduduks.id') // Menggunakan format tabel.kolom
                ->select('kks.id as no_kk', 'kks.kepala_kk as kepala_kk', 'penduduks.nama_lengkap as nama_lengkap')
                ->get();
        return view('admin.kependudukan.penduduk.index',[
            'level' => $level,
            'profil' => $profil,
            'refsks' => $refsks,
            'kks' => $kks,
        ])->with('tittle', 'Data Penduduk');
    }

    public function get_penduduk(Request $request)
    {
        if ($request->ajax()) {
            $penduduks = Penduduk::query()->orderBy('created_at', 'desc')->get();

            return DataTables::of($penduduks)
                ->addIndexColumn()
                ->addColumn('opsi', function ($row) {
                    return '
                                <a href="javascript:void(0)" data-id="'.$row->id.'" class="btn btn-icon btn-sm btn-outline btn-outline-info text-hover-white mb-1 edit-penduduk">
                                    <i class="bi bi-pencil text-info fs-4" data-bs-toggle="tooltip" data-bs-custom-class="tooltip-dark" data-bs-placement="top" title="Update"></i>
                                </a>
                                <a href="javascript:void(0)" data-id="'.$row->id.'" class="btn btn-icon btn-sm btn-outline btn-outline-danger text-hover-white mb-1 delete-penduduk">
                                    <i class="bi bi-trash text-danger fs-4" data-bs-toggle="tooltip" title="Hapus"></i>
                                </a>
                            ';
                })
                ->addColumn('tgl_lahir_indo', function($row) {
                    return $row->tempat_lahir . ',<br>' . date('d-m-Y', strtotime($row->tgl_lahir));
                })
                ->addColumn('jenis_kelamin', function($row) {
                    if($row->jenis_kelamin=="L"){
                        $jenis_kelamin = "LAKI-LAKI";
                    }else{
                        $jenis_kelamin = "PEREMPUAN";
                    }
                    return $jenis_kelamin;
                })
                ->addColumn('status_perkawinan', function($row) {
                    if($row->status_perkawinan=="B"){
                        $status_perkawinan = "BELUM KAWIN";
                    }elseif($row->status_perkawinan=="S"){
                        $status_perkawinan = "SUDAH KAWIN";
                    }else{
                        $status_perkawinan = "PERNAH KAWIN";
                    }
                    return $status_perkawinan;
                })
                ->addColumn('rt_rw', function($row) {
                    return $row->rt.'/'.$row->rw;
                })
                ->rawColumns(['opsi','tgl_lahir_indo','jenis_kelamin','status_perkawinan','rt_rw'])
                ->make(true);
        }
    }

    public function store(Request $request)
    {
        $request->validate([
            'id'                => 'required|unique:penduduks,id|max:16',
            'no_kk'             => 'required|exists:kks,id',
            'nama_lengkap'      => 'required|string|max:50',
            'tempat_lahir'      => 'required|string|max:30',
            'tgl_lahir'         => 'required',
            'jenis_kelamin'     => 'required',
            'agama'             => 'required',
            'status_perkawinan' => 'required',
            'dusun'             => 'required',
            'rw'                => 'required',
            'rt'                => 'required',
        ]);
        $input = $request->all();
        $input['tempat_lahir'] = strtoupper($request->tempat_lahir);
        $input['dusun'] = strtoupper($request->dusun);
        $simpan = Penduduk::create($input);
        if ($simpan) {
            Alert::success('Berhasil', 'Data KK berhasil ditambahkan.');
        }else {
            Alert::error('Gagal', 'Data KK gagal ditambahkan.');
        }

        return redirect()->route('admin-kependudukan-penduduk');
    }

    public function getPendudukId($id)
    {
        $penduduk = Penduduk::findOrFail($id);
        return response()->json($penduduk);
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'id'                => 'required|max:16|unique:penduduks,id,' . $id,
            'no_kk'             => 'required|exists:kks,id',
            'nama_lengkap'      => 'required|string|max:50',
            'tempat_lahir'      => 'required|string|max:30',
            'tgl_lahir'         => 'required',
            'jenis_kelamin'     => 'required',
            'agama'             => 'required',
            'status_perkawinan' => 'required',
            'dusun'             => 'required',
            'rw'                => 'required',
            'rt'                => 'required',
        ]);
        $penduduk = Penduduk::findOrFail($id);
        $simpan = $penduduk->update([
            'id'                => $request->id,
            'no_kk'             => $request->no_kk,
            'nama_lengkap'      => $request->nama_lengkap,
            'tempat_lahir'      => $request->tempat_lahir,
            'tgl_lahir'         => $request->tgl_lahir,
            'jenis_kelamin'     => $request->jenis_kelamin,
            'agama'             => $request->agama,
            'status_perkawinan' => $request->status_perkawinan,
            'dusun'             => $request->dusun,
            'rw'                => $request->rw,
            'rt'                => $request->rt,
        ]);
        if ($simpan) {
            Alert::success('Berhasil', 'Data Penduduk berhasil diperbarui.');
        }else {
            Alert::error('Gagal', 'Data Penduduk gagal diperbarui.');
        }

        return redirect()->route('admin-kependudukan-penduduk');
    }

    public function delete($id)
    {
        $penduduk = Penduduk::findOrFail($id);
        $simpan = $penduduk->delete();

        if ($simpan) {
            Alert::success('Berhasil', 'Data Penduduk berhasil dihapus!');
        }else {
            Alert::error('Gagal', 'Data Penduduk gagal dihapus!');
        }

        return redirect()->route('admin-kependudukan-penduduk');
    }
}
